/**
 *
 * Arduino skeleton sketch for controlling the CameraPro app for Nokia Symbian. For detailed API instructions
 * and examples check the manuals on http://www.tequnique.com/camerapro#netmodule
 *
 * CameraPro website: http://www.tequnique.com/camerapro
 * Author: Harald Meyer, 2013
 *
 */

#include <EasyTransfer.h>  // https://github.com/madsci1016/Arduino-EasyTransfer
#include <SoftwareSerial.h>


// Pin used for capture button
#define CAPTURE_BTN_PIN 5

// EasyTransfer objects for in- and out communication
EasyTransfer _etIn, _etOut; 

// Structures for sending and receiving data in a fixed format
struct RECEIVE_DATA_STRUCTURE {
  int8_t command;
};

struct SEND_DATA_STRUCTURE {
  int8_t command;
  short value;
};

RECEIVE_DATA_STRUCTURE _rxData;
SEND_DATA_STRUCTURE _txData;

// Commands
#define MODE_UNKOWN              0               // Unkown state (default)
#define MODE_CAMERAPRO           1               // This command is sent by CameraPro when it is ready to receive commands.

#define CAMERAPRO_COMMAND_OK	20
#define CAMERAPRO_COMMAND_FAILED	21
#define CAMERAPRO_CAMERA_CONTROL	23       // Only supported for Wifi
#define CAMERAPRO_CAMERA_MODE	24
#define CAMERAPRO_GET_CAPTURED_IMAGE	25
#define CAMERAPRO_EFFECT_MODE	26
#define CAMERAPRO_EXPOSURE_COMPENSATION	27
#define CAMERAPRO_EXPOSURE_MODE	28
#define CAMERAPRO_FLASH_MODE	29
#define CAMERAPRO_FOCUS_MODE	30
#define CAMERAPRO_FOCUS_POSITION_X	31
#define CAMERAPRO_FOCUS_POSITION_Y	32
#define CAMERAPRO_IMAGECONTROLS_BRIGHTNESS	33
#define CAMERAPRO_IMAGECONTROLS_CONTRAST	34
#define CAMERAPRO_IMAGECONTROLS_SATURATION	35
#define CAMERAPRO_IMAGECONTROLS_SHARPNESS	36
#define CAMERAPRO_ISO_VALUE	37
#define CAMERAPRO_NDFILTER_MODE	38
#define CAMERAPRO_RESOLUTION	39
#define CAMERAPRO_SWITCH_CAMERA	40
#define CAMERAPRO_WHITEBALANCE_MODE	41
#define CAMERAPRO_ZOOM	42

int8_t _currentMode = MODE_UNKOWN;


//
// Mode handler functions
//
void handleCameraProFocusMode()
{
  // Check capture button state
  int buttonState = digitalRead(CAPTURE_BTN_PIN);

  // If it's pressed then send a capture command
  if (buttonState == HIGH) 
  {     
    _txData.command = CAMERAPRO_CAMERA_CONTROL;  // The command.
    _txData.value = 0;                           // The value for triggering image capture as described in the API manual.
    _etOut.sendData();
    Serial.flush();
    delay(1000);
  }
  
  //TODO: other commands can be implemented here
}

//
// Arduino functions
//

// Arduino setup function
void setup() 
{
  // Open USB connection (for debugging)
  Serial.begin(9600);
  Serial.println("USB/Serial connected");
  
  // Set reference to BT connection
  _etIn.begin(details(_rxData), &Serial);
  _etOut.begin(details(_txData), &Serial);
}

// Arduino looper
void loop() 
{    
  // Read next command
  if (_etIn.receiveData())
  {
    if (_rxData.command < CAMERAPRO_COMMAND_OK)
    {
      if (_currentMode != _rxData.command)
      {
        _currentMode = _rxData.command;
        
        // Setup mode specific stuff
        if (_currentMode == MODE_CAMERAPRO)
        {
          // Capture button
          pinMode(CAPTURE_BTN_PIN, INPUT); 
         
          // Reply with "ok"
          _txData.command = CAMERAPRO_COMMAND_OK;
          _txData.value = 0;
          _etOut.sendData();
          Serial.flush();
          delay(30);
        }
      }
    }
  }
  else
  {
    // Run mode dependent function
    switch(_currentMode)
    {
      case MODE_CAMERAPRO:
        handleCameraProFocusMode();
      break;
      default:
      // do nothing
      break;
    }
  }
}















